import com.probertson.xmlrpc.PendingCall;
import com.probertson.xmlrpc.ServiceBase;

import mx.rpc.Responder;

/**
 * Essentially represents a single remote method of a service, and acts as a go-between for
 * {@see com.probertson.xmlrpc.Service} instances and {@see com.probertson.xmlrpc.PendingCall}
 * instances, to keep a reference to a customer Responder. Instances are created by the Service
 * class and generally should not be created directly.
 * 
 * @access	internal
 * 
 * @author H. Paul Robertson
 */
class com.probertson.xmlrpc.ServiceMethod extends Object
{
	#include "XmlRpcComponentVersion.as"

	//
	// Private Fields
	//
	private var _service:ServiceBase;
	private var _responder:Responder;
	private var _methodName:String;


	//
	// Constructor
	//
	/**
	 * Creates a new ServiceMethod instance.
	 * 
	 * @access	internal
	 * 
	 * @param	methodName	The name of the remote method to call.
	 * @param	parent		The ServiceBase which is calling the method.
	 */
	public function ServiceMethod(methodName:String, parent:ServiceBase)
	{
		super();
		_methodName = methodName;
		_service = parent;
		_responder = parent.responder;
	}


	//
	// Public Methods
	//
	/**
	 * Forwards a remote method call from the Service on to the Connection.
	 * 
	 * @access	internal
	 * 
	 * @param	args	Contains the arguments to be passed to the remote method call.
	 */
	public function call(args:Array):PendingCall
	{
		//_service.log.logInfo("Calling " + _service.name + "." + _methodName);
		var result:PendingCall = new PendingCall(_service, _methodName);
		result.responder = _responder;

		// generate the arguments to pass to the XMLRPC method
		var callArgs:Array;
		if (undefined == args || null == args)
		{
			callArgs = new Array();
		}
		else
		{
			callArgs = args.concat();
		}

		// get the full method name for the XMLRPC call
		var fullMethodName:String = getFullMethodName(_service.name, _methodName);

		// send the call through the connection
		_service.connection.call(fullMethodName, result, callArgs);

		return result;
	}


	//
	// Private Methods
	//
	private static function getFullMethodName(serviceName:String, methodName:String):String
	{
		var result:String = "";
		if (serviceName != null && serviceName != "")
		{
			result += serviceName;
			result += ".";
		}
		result += methodName;
		return result;
	}
}