import com.probertson.xmlrpc.Connection;
import com.probertson.xmlrpc.PendingCall;
import com.probertson.xmlrpc.ServiceMethod;

import mx.rpc.Responder;
//import mx.services.Log;

//import com.probertson.ArgumentNullException;

/**
 * Base (super) class for XML-RPC services.
 * This class is intended to be an abstract class. Instances of ServiceBase should not be created directly;
 * instead, you can use the dynamic {@see com.probertson.xmlrpc.Service} or create your own strongly typed ServiceBase subclass
 * in order to access an XML-RPC service.
 * 
 * Implementers should define a constructor which calls ServiceBase's constructor with all arguments.
 * Implementers can define their own methods to serve as wrappers for the methods of the specific
 * XMLRPC service they are accessing.  Those methods should in turn call the {@link com.probertson.xmlrpc.ServiceBase#call} method
 * to actually make the XML-RPC method call.
 * 
 * @access abstract
 * 
 * @author H. Paul Robertson
 */
class com.probertson.xmlrpc.ServiceBase extends Object
{
	#include "XmlRpcComponentVersion.as"

	//
	// Private Fields
	//
	private var _conn:Connection;
	private var _serviceName:String;
	private var _responder:Responder;
	//public var log:Log;

	// key/value dictionary of ServiceMethod instances
	private var _serviceMethods:Object = new Object();


	//
	// Constructor
	//
	/**
	 * Creates a new ServiceBase instance.
	 * @access	protected
	 * 
	 * @param	serviceUri		The uri of the XMLRPC service to be called. Use null if
	 * 							a Connection instance is being provided.
	 * 											
	 * @param	serviceName		The name of the XMLRPC service. This is not official terminology
	 * 							but most XMLRPC methods are named "serviceName.methodName", often
	 * 							with several methods having the same serviceName (e.g. math.Add,
	 * 							math.Subtract). The first part of the method names (e.g. "math"), with no
	 * 							period ".", should be provided as the serviceName parameter.
	 * 							If the methods provided at a given url do not follow the convention
	 * 							of using a service name prefix, pass null instead.
	 * 											
	 * @param	conn			A {@see com.probertson.xmlrpc.Connection} instance that should be used by
	 * 							this service.  This can be given as an alternative to providing a
	 * 							serviceUri.  In practice there is little reason to use this parameter;
	 * 							Connections with the same uri are reused internally in any case, so there
	 * 							is no performance benefit from passing in an already-created Connection.
	 * 											
	 * @param	resp			A mx.rpc.Responder instance that should receive Result and Fault responses
	 * 							from the method calls. You can use this parameter if you want a single
	 * 							Responder to handle responses from all remote method calls, rather than
	 * 							creating a separate Responder for each method.
	 */
	public function ServiceBase(serviceUri:String, serviceName:String, conn:Connection, resp:Responder)
	{
		super();

		if ((undefined == serviceUri || null == serviceUri || "" == serviceUri) && (undefined == conn || null == conn))
		{
			//throw new ArgumentNullException("You must define either a service URI or a Connection");
			throw new Error("You must either define a service URI or pass a Connection reference.");
		}

		//log = logger;

		//log.logInfo("Creating Service for " + serviceName, Log.VERBOSE);
		// use the passed connection, or create a new connection
		if (null == conn)
		{
			conn = Connection.getConnection(serviceUri);
			if (null == conn)
			{
				// log.logInfo("Creating gateway connection for " + serviceUri, Log.VERBOSE);
				conn = Connection.createConnection(serviceUri/*, log */);
			}
		}
		_conn = conn;
		_serviceName = ("" == serviceName) ? null : serviceName;
		_responder = resp;

		//log.logInfo("Successfully created Service", Log.VERBOSE);
	}


	//
	// Public Properties
	//
	/**
	 * Gets the {@see com.probertson.xmlrpc.Connection} instance used by this instance.
	 */
	public function get connection():Connection
	{
		return _conn;
	}

	/**
	 * Gets the name of the remote service which this instance "wraps."
	 */
	public function get name():String
	{
		return _serviceName;
	}

	/**
	 * Gets the mx.rpc.Responder assigned to this instance, if one was assigned to it.
	 */
	public function get responder():Responder
	{
		return _responder;
	}


	//
	// "Protected" methods -- intended to be called by implementers
	//
	/**
	 * Implementers should create their own mechanism for allowing access to the XML-RPC
	 * service's methods (e.g. a wrapper method). Your subclass should call
	 * this method to actually perform the remote method call.
	 * this method.
	 * @access	protected
	 * 
	 * @param	methodName	The name of the remote method to call.
	 * 
	 * @param	[param1...paramN]	The parameters to be passed to the remote method. These can be any of
	 * 								the allowed XML-RPC types.
	 */
	private function call(methodName:String):PendingCall
	{
		// get additional arguments that were passed in
		var args:Array = null;
		if (arguments.length > 1)
		{
			args = arguments.slice(1);
		}
		var serviceMethod:ServiceMethod = getServiceMethod(methodName);
		return serviceMethod.call(args);
	}

	/**
	 * Implementers can use this method to get access to the {see com.probertson.xmlrpc.ServiceMethod} instance which will handle
	 * a particular remote method call in order to make the remote call through the RemoteMethod instance
	 * rather than through {link com.probertson.xmlrpc.ServiceBase#call}.  For most cases ServiceBase.call() will suffice.
	 * The most common reason you might want to get access to the ServiceMethod would be to pass the
	 * method parameters as an array rather than as separate parameters.
	 * @access	protected
	 * 
	 * @param	methodName	The name of the remote method whose respective ServiceMethod instance
	 * 						should be returned.
	 */
	private function getServiceMethod(methodName:String):ServiceMethod
	{
		// look in the dictionary to see if we already have a ServiceMethod object
		// for this method
		var result:ServiceMethod = _serviceMethods[methodName];
		if (undefined == result || null == result)
		{
			result = new ServiceMethod(methodName, this);
			_serviceMethods[methodName] = result;
		}
		return result;
	}
}