import com.probertson.xmlrpc.Connection;
import com.probertson.xmlrpc.ServiceBase;
import com.probertson.xmlrpc.ServiceMethod;

import mx.rpc.Responder;


/**
 * Represents an XML-RPC service which provides one or more remote methods. This is the main
 * class used by end-developers to make remote method calls using XML-RPC. Developers should
 * create a Service instance, then call methods on the Service instance with names matching
 * the available remote methods.
 * @example If an XML-RPC service named "animals" provides a method named "getSound"
 * (which would probably be commonly referred to as "animals.getSound"), the following lines
 * of code could be used to call the method and retrieve the result:
 * <pre>
 * var animalService:com.probertson.xmlrpc.Service = new com.probertson.xmlrpc.Service("http://example.com/serviceURL", "animals", null, null);
 * var animalCall:mx.rpc.PendingCall = animalService.getSound("dog");
 * // assume other functions named "getSoundResult" and "getSoundFault" are defined.
 * // those functions will be called with the result (or if an error happens) respectively.
 * animalCall.responder = new com.probertson.rpc.RelayResponder2(this, getSoundResult, getSoundFault);
 * </pre>
 * 
 * @author H. Paul Robertson
 */
dynamic class com.probertson.xmlrpc.Service extends ServiceBase
{
	#include "XmlRpcComponentVersion.as"


	//
	// Constructor
	//
	/**
	 * Creates a new Service instance.
	 * 
	 * @param	serviceUri	The fully qualified (including http/https) uri of the service. This
	 * 						parameter is required unless a Connection is provided to the "conn"
	 * 						parameter, in which case null should be passed.
	 * @param	serviceName	The name of the XML-RPC service. Typically XML-RPC methods are referred
	 * 						to with a two-part name (e.g. "math.sum", "math.difference"). The service
	 * 						name is the first part of the method name, up to but not including the
	 * 						period (e.g. "math").  If a remote method has a simple method name (i.e.
	 * 						without a period) just pass null for this parameter, and any methods
	 * 						at the service uri with no "service name" will be available to this instance.
	 * @param	conn		A {@link com.probertson.xmlrpc.Connection} instance (which will
	 * 						have a service uri already defined) to be used by this instance.
	 * 						Under most circumstances this parameter should be passed null.
	 * @param	resp		A mx.rpc.Responder instance specifying result and fault methods
	 * 						to be used for all remote method calls through this service.
	 * 						Generally, those are specified through the {@link com.probertson.xmlrpc.PendingCall#responder}
	 * 						property of the {@link com.probertson.xmlrpc.PendingCall} instance returned
	 * 						by the Service method which wraps the remote call, so this paramater
	 * 						will commonly be passed null.
	 */
	public function Service(serviceUri:String, serviceName:String, conn:Connection, resp:Responder)
	{
		super(serviceUri, serviceName, conn, resp);
	}


	//
	// Private methods
	//
	private function _createServiceMethod(methodName:String):Function
	{
		// rather than use the superclass's call method, we have to call the ServiceMethod
		// directly because we are passing the arguments in an array rather
		// than separately (which is what ServiceBase.call() is designed to take.
		// However, by using super.getServiceMethod(), we will still be able to reuse the ServiceMethod
		// instance to conserve resources.
		var serviceMethod:ServiceMethod = super.getServiceMethod(methodName);

		// The function generated here will be accessed as <serviceVar>.methodName(args), via __resolve
		var result:Function = function()
		{
			return serviceMethod.call(arguments);
		}
		return result;
	}

	private function __resolve(methodName:String):Function
	{
		// Trick borrowed from Macromedia (mx.remoting.Service)
		// Register the function as a property on the Service so that __resolve isn't called again.
		var newMethod:Function = _createServiceMethod(methodName);
		this[methodName] = newMethod;
		return newMethod;
	}
}