import com.probertson.xmlrpc.IXmlRpcStruct;
import com.probertson.xmlrpc.XmlRpcFault;

/**
 * Performs the task of serializing method calls into XML-RPC format, and deserializing result
 * XML-RPC messages into ActionScript objects. This class is used internally, and is not
 * intended to be called directly.
 * 
 * @access	internal
 * 
 * @author H. Paul Robertson
 */
class com.probertson.xmlrpc.Serializer extends Object
{
	#include "XmlRpcComponentVersion.as"

	//
	// Constructor
	//
	private function Serializer()
	{
	}


	//
	// Singleton Implementation
	//
	private static var _instance:Serializer;

	/**
	 * Retrieves the Singleton instance of the Serializer class.
	 * @access	internal
	 * 
	 * @return	the single instance of the Serializer class.
	 */
	public static function getInstance():Serializer
	{
		if (null == _instance)
		{
			_instance = new Serializer();
		}
		return _instance;
	}


	//
	// Serializing Request
	//
	/**
	 * Serializes (converts) an ActionScript method call into an XML-RPC method call.
	 * @access	internal
	 * 
	 * @param	methodName	The name of the remote method being called.
	 * @param	args		The arguments being passed to the remote method.
	 */
	public function serializeRequest(methodName:String, args:Array):XML
	{
		var result:XML = new XML();
		result.ignoreWhite = true;
		result.contentType = "text/xml";
		result.xmlDecl = "<?xml version=\"1.0\" ?>";

		// build the main nodes
		// "<methodCall>" root node
		var rootNode:XMLNode = result.createElement("methodCall");
		result.appendChild(rootNode);
		// "<methodName>" node
		var methodNode:XMLNode = result.createElement("methodName");
		methodNode.appendChild(result.createTextNode(methodName));
		rootNode.appendChild(methodNode);

		// add the parameter nodes
		// "<params>" node
		var paramsNode:XMLNode = result.createElement("params");
		rootNode.appendChild(paramsNode);

		if (args != undefined && args != null)
		{
			var oneParamNode:XMLNode;
			var numParams:Number = args.length;
			for (var i:Number = 0; i < numParams; i++)
			{
				oneParamNode = result.createElement("param");
				oneParamNode.appendChild(createParamNode(result, args[i]));
				paramsNode.appendChild(oneParamNode);
			}
		}

		return result;
	}


	//
	// Private Methods (serializing)
	//
	private function createParamNode(xml:XML, arg:Object):XMLNode
	{
		var valueNode:XMLNode = xml.createElement("value");
		var typeNode:XMLNode;

		if (typeof(arg) == "string" || arg instanceof String)
		{
			var strVal:String = arg.toString();
			// if it contains unescaped entity values, flag it as CDATA
			if (needsCData(strVal))
			{
				strVal = "<![CDATA[" + strVal + "]]>";
			}
			typeNode = xml.createElement("string");
			typeNode.appendChild(xml.createTextNode(strVal));
		}
		else if (typeof(arg) == "boolean" || arg instanceof Boolean)
		{
			typeNode = xml.createElement("boolean");
			typeNode.appendChild(xml.createTextNode(arg.toString()));
		}
		else if (typeof(arg) == "number" || arg instanceof Number)
		{
			var numVal:Number = Number(arg);
			// determine if it's an int or a double
			if (Math.floor(numVal) == numVal)
			{
				typeNode = xml.createElement("i4");
			}
			else
			{
				typeNode = xml.createElement("double");
			}
			typeNode.appendChild(xml.createTextNode(numVal.toString()));
		}
		else if (arg instanceof Date)
		{
			// non-type-checked conversion of untyped date to typed date 
			// uses dynamic access to methods because there isn't a way to cast arg to Date
			var dateVal:Date = new Date(arg["getFullYear"](), arg["getMonth"](), arg["getDate"](), arg["getHours"](), arg["getMinutes"](), arg["getSeconds"]());
			typeNode = xml.createElement("dateTime.iso8601");
			typeNode.appendChild(xml.createTextNode(formatDateTimeISO8601(dateVal)));
		}
		else if (arg instanceof Array)
		{
			typeNode = xml.createElement("array");
			var dataNode:XMLNode = xml.createElement("data");
			var arrayVal:Array = Array(arg);
			var numElements:Number = arrayVal.length;
			for (var i:Number = 0; i < numElements; i++)
			{
				dataNode.appendChild(createParamNode(xml, arrayVal[i]));
			}
			typeNode.appendChild(dataNode);
		}
		else if (arg instanceof IXmlRpcStruct)
		{
			typeNode = xml.createElement("struct");
			var structVal:IXmlRpcStruct = IXmlRpcStruct(arg);
			var members:Array = structVal.getStructProperties();
			var numMembers:Number = members.length;
			// loop through struct members, adding <member> nodes
			// and <name> and <value> nodes for each member.
			for (var i:Number = 0; i < numMembers; i++)
			{
				var memberName:String = String(members[i]);
				var memberNode:XMLNode = xml.createElement("member");
				// add the <name> node for this member
				var nameNode:XMLNode = xml.createElement("name");
				nameNode.appendChild(xml.createTextNode(memberName));
				memberNode.appendChild(nameNode);
				// add the <value> node, using the value in the struct member
				// This uses the [] property access operator, since that is the most convenient
				// way to dynamically access object members
				memberNode.appendChild(createParamNode(xml, structVal[memberName]));
				typeNode.appendChild(memberNode);
			}
		}
		valueNode.appendChild(typeNode);
		return valueNode;
	}

	private function formatDateTimeISO8601(date:Date):String
	{
		var result:String;
		result = date.getFullYear().toString();
		result += zeroPadNumber((date.getMonth() + 1), 2);
		result += zeroPadNumber(date.getDate(), 2);
		result += "T";
		result += zeroPadNumber(date.getHours(), 2);
		result += ":";
		result += zeroPadNumber(date.getMinutes(), 2);
		result += ":";
		result += zeroPadNumber(date.getSeconds(), 2);
		return result;
	}

	private function zeroPadNumber(num:Number, minLength:Number):String
	{
		var result:String = num.toString();
		while (result.length < minLength)
		{
			result = "0" + result;
		}
		return result;
	}

	private function needsCData(str:String):Boolean
	{
		var result:Boolean = false;
		if (str.indexOf("<", 0) >= 0)
		{
			result = true;
		}
		else if (str.indexOf("\"", 0) >= 0)
		{
			result = true;
		}
		else
		{
			var pos:Number = str.indexOf("&", 0);
			while (pos != -1)
			{
				if (pos != str.indexOf("&amp;", pos) && pos != str.indexOf("&lt;", pos) && pos != str.indexOf("&gt;", pos) && pos != str.indexOf("&quote;", pos))
				{
					result = true;
					break;
				}
				pos = str.indexOf("&", pos + 1);
			}
		}
		return result;
	}


	//
	// Deserialize Response
	//
	/**
	 * Converts the XML-RPC message returned by a remote method call into an ActionScript
	 * object.
	 * @access	internal
	 * 
	 * @param	response	The XML returned by the remote method.
	 */
	public function deserializeResponse(response:XML):Object
	{
		// the deserialization depends on having ignoreWhite set to true
		// if it isn't, we have to set it and re-parse the xml
		if (!response.ignoreWhite)
		{
			response.ignoreWhite = true;
			response.parseXML(response.toString());
		}
		return deserializeNode(response.firstChild);
	}

	private function deserializeNode(node:XMLNode, value:Object):Object
	{
		for (var currNode:XMLNode = node; currNode != null; currNode = currNode.nextSibling)
		{
			switch (currNode.nodeName)
			{
				case "array":
					return deserializeNode(currNode.firstChild, new Array());
				case "struct":
					return deserializeNode(currNode.firstChild, new Object());
				case "member":
					value[currNode.firstChild.firstChild.nodeValue] = deserializeNode(currNode.firstChild.nextSibling);
					break;
				case "data":
					for (var arrayItemNode:XMLNode = currNode.firstChild; arrayItemNode != null; arrayItemNode = arrayItemNode.nextSibling)
					{
						value.push(deserializeNode(arrayItemNode, value));
					}
					return value;
				case "value":
					switch (currNode.firstChild.nodeName)
					{
						case "i4":
						case "int":
							return parseInt(currNode.firstChild.firstChild.nodeValue);
						case "double":
							return parseFloat(currNode.firstChild.firstChild.nodeValue);
						case "string":
							return currNode.firstChild.firstChild.nodeValue;
						case null: // if it's null, they omitted the data type node, so we use the default (string)
							return currNode.firstChild.nodeValue;
						case "boolean":
							return ("1" == currNode.firstChild.firstChild.nodeValue);
						case "dateTime.iso8601":
							var dateStr:String = currNode.firstChild.firstChild.nodeValue;
							var year:Number = parseInt(dateStr.substring(0, 4));
							var month:Number = parseInt(dateStr.substring(4, 6)) - 1;
							var day:Number = parseInt(dateStr.substring(6, 8));
							var hour:Number = parseInt(dateStr.substring(9, 11));
							var minute:Number = parseInt(dateStr.substring(12, 14));
							var second:Number = parseInt(dateStr.substring(15, 17));
							return new Date(year, month, day, hour, minute, second);
						default:
							return deserializeNode(currNode.firstChild, value);
					}
					break;
				case "fault":
					var faultStruct:Object = deserializeNode(currNode.firstChild.firstChild);
					return new XmlRpcFault(faultStruct.faultCode, faultStruct.faultString);
				case null : // this is here just in case we miss the value for some reason.
					return currNode.nodeValue;
				default : // this is here just in case some unexpected node name comes through for some reason
					return deserializeNode(currNode.firstChild, value);
			}
		}
		return value;
	}
}