import com.probertson.xmlrpc.PendingCall;
import com.probertson.xmlrpc.Serializer;
import com.probertson.xmlrpc.XmlRpcFault;

//import com.probertson.ArgumentNullException;
import com.probertson.utils.UrlUtility;

/**
 * Manages network communication for an XML-RPC {@see com.probertson.xmlrpc.Service}.
 * In general Connections will be managed internally. Multiple Services which access the
 * same url will reuse Connection instances, so there is no need to manually force
 * Connections to be reused.
 * 
 * @access	internal
 * @author	H. Paul Robertson
 */
class com.probertson.xmlrpc.Connection extends Object
{
	#include "XmlRpcComponentVersion.as"

	//
	// Private Fields
	//
	// key/value dictionary of connections
	private static var _connections:Object;

//	private var log:Log;
	private var _uri:String;


	//
	// Constructor
	//
	private function Connection(uri:String)
	{
		super();
		_uri = uri;
	}


	//
	// Public Methods
	//
	/**
	 * Makes an XMLRPC method call. This should not be called by developers; instead,
	 * create a {@link com.probertson.xmlrpc.Service} instance and call remote methods through that.
	 * @access	internal
	 * 
	 * @param	methodName		Full method name of the XML-RPC method to call.
	 * @param	pendingCall		Instance which is listening for the response from the call.
	 * @param	args			Arguments to pass to the XML-RPC method call.
	 */
	public function call(methodName:String, pendingCall:PendingCall, args:Array):Void
	{
		// create the XML document representing the method call
		var serializer:Serializer = Serializer.getInstance();
		var methodCall:XML = serializer.serializeRequest(methodName, args);
		var responseXML:XML = new XML();
		responseXML.ignoreWhite = true;

		// set up the listener to handle the return
		responseXML.onLoad = function(success:Boolean)
		{
			if (success)
			{
				// parse the response xml
				var deserializer:Serializer = Serializer.getInstance();
				// within this XML.onLoad function, "this" refers to the XML object
				var result:Object = deserializer.deserializeResponse(this);
				// if it's a fault, send it to pendingCall.onStatus;
				// if it's a result, send it to pendingCall.onResult
				if (result instanceof XmlRpcFault)
				{
					pendingCall.onStatus(XmlRpcFault(result));
				}
				else
				{
					pendingCall.onResult(result);
				}
			}
			else
			{
				// send an error message to pendingCall.onStatus
				pendingCall.onStatus(new XmlRpcFault(404, "The Connection URL was not found (HTTP: Status 404)"));
			}
			// clean up
			delete methodCall;
			delete responseXML;
		};

		// make the call
		methodCall.sendAndLoad(_uri, responseXML);
	}


	//
	// Public Static Methods
	//
	/**
	 * Creates a new connection.
	 * @access	internal
	 * 
	 * @param	uri		The uri of the connection.
	 * 
	 * @return	The newly created connection.
	 */
	public static function createConnection(uri:String):Connection
	{
		if (undefined == uri || null == uri || "" == uri )
		{
			//throw new ArgumentNullException("uri");
			throw new Error("The parameter 'uri' cannot contain a null reference");
		}
		// convert the uri to a fully qualified uri including protocol etc.
		UrlUtility.getHttpUrl(uri);
		var result:Connection = new Connection(uri);
		_connections[uri] = result;

		return result;
	}

	/**
	 * Obtain a reference to a Connection, if one has already been created for a particular uri.
	 * @access	internal
	 * 
	 * @param	uri		The uri of the connection.
	 * 
	 * @return	The connection with that uri.
	 */
	public static function getConnection(uri:String):Connection
	{
		var result:Object = _connections[uri];
		if (result != undefined && result != null && (result instanceof Connection))
		{
			return Connection(result);
		}
		return null;
	}
}